/** @file   gameobject.cpp
 * @brief   Implementation of GameObject - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#include "gameobject.h"
#include "www_map.h"
#include "consts.h"
#include "door.h"
#include "warglobals.h"
#include "redrawqueue.h"
#include "www_assert.h"
#include "basecontroller.h"
#include "objectmessage.h"
#include "datatypes.h"

using namespace eng2d;

namespace WeWantWar {

//********************************************************************
//                                                                   *
//      Static members and constants                                 *
//                                                                   *
//********************************************************************

/** Number of extra counters the instances of GameObject-class has. */
const int GameObject::NUM_OF_COUNTERS;
/** Number of collision points the GameObjects have. These are the
 * first points found from 'm_ctrlPoints'-vector.
 */
const int GameObject::NUM_OF_COLL_POINTS;

const GameObject::BProp GameObject::PROP_GRENADEFORCE;
const GameObject::BProp GameObject::PROP_SHOTPARALYSES;

const GameObject::IDCode GameObject::DEFAULT_ID;

int GameObject::s_numberOfGameObjects = 0;


//********************************************************************
//                                                                   *
//      Constructors, destructor and operators                       *
//                                                                   *
//********************************************************************

/** Constructs new gameobject
 */
GameObject::GameObject( ) :
  AnimatedObject(),
  
  m_hidden( false ),
  m_health( 100 ),
  m_armor( 1 ),
  m_propertyFlags( 0 ),
  
  m_state( STATE_LIVING ),
  m_objectID( DEFAULT_ID ),
  m_pController( 0 )
{
  // Reset the counter-array.
  m_counters[0] = -1;
  m_counters[1] = -1;
  m_counters[2] = -1;
  
  
  // Add the collision control points.
  for (int i=0; i < NUM_OF_COLL_POINTS; i++) {
    this->addCollisionPoint( Vec2D(0, 0) );
  }
  
  // Increase the objectcount.
  s_numberOfGameObjects += 1;
}



/** Destructor
 */
GameObject::~GameObject()
{
  delete m_pController;
  
  // Decrease the objectcount.
  s_numberOfGameObjects -= 1;
}



//********************************************************************
//                                                                   *
//      Public interface                                             *
//                                                                   *
//********************************************************************


/** Sets the target. Default implementation does nothing.
 */
void GameObject::target( const Vec2D& t )
{
}



/** Sets the controlpoints
 */
void GameObject::setCtrlPoint(int pnum, const Vec2D& p)
{
  WWW_ASSERT( pnum >= 0 );
  WWW_ASSERT( pnum < m_ctrlPoints.size() );
  
  m_ctrlPoints.at(pnum) = p;
}



/** Adds the given controlpoint
 */
void GameObject::addCtrlPoint(const Vec2D& p)
{
  m_ctrlPoints.push_back( p );
}



/** Sets the hidden flag
 */
void GameObject::hidden( bool h )
{
  m_hidden = h;
}




/** Sets the value for the counter
 */
void GameObject::setCounter( int cnum, int val )
{
  m_counters[cnum] = val;
}




/** The update method. Update the extra counters, animations and handles
 * the force-vector.
 */
void GameObject::updateObject()
{
  // First we update the derived classes.
  this->update();
  
  
  
  // Update the extra counters.
  for (int i=0; i < NUM_OF_COUNTERS; i++) {
    m_counters[i] -= 1;
  }
  
  // Update the movement
  this->updateMovement();
  
  // Update the current animation.
  this->updateAnim();
  
  // And kill us if we've run out of health.
  if ( m_state == STATE_LIVING ) {
    if ( m_health < 0 ) {
      this->kill();
    }
  }
}



/** Causes the given amount of damage. The damage gets first divided be
 * armor-level and then it's substracted from the health.
 */
bool GameObject::causeDamage(int points)
{
  this->makeSound( SND_PAIN );
  float dam = static_cast<float>(points) / m_armor;
  
  m_health -= dam;
  if ( static_cast<int>(m_health) < 1 ) {
    return true;
  }
  return false;
}



/** Another version of cause damage.
 */
bool GameObject::causeDamage(Bullet* pB)
{
  if ( pB->iType == Bullet::EMinigun ) {
    if ( this->hasProperty( PROP_GRENADEFORCE ) ) {
      Vec2D f( pB->velocity() );
      f.scale( 0.4 );
      this->setForce( f );
    }
  }
  if ( pB->iType != Bullet::EFlameThrower ) {
    return this->causeDamage( static_cast<int>( pB->iDamage ) );
  }
  
  // The flamethrower is handled differently
  if ( (rand() % 10) == 1 ) {
    this->makeSound( SND_PAIN );
  }
  
  float dam = static_cast<float>( pB->iDamage ) / m_armor;
  m_health -= dam;
  if ( static_cast<int>(m_health) < 1 ) {
    return true;
  }
  return false;
}



/** Sets the state
 */
void GameObject::state( State s )
{
  m_state = s;
}



/** Sets the health
 */
void GameObject::setHealth( int h )
{
  m_health = h;
}



/** Sets the armor
 */
void GameObject::setArmor( float a )
{
  m_armor = a;
}



/** Sets the properties.
 */
void GameObject::setProperties( BProp p )
{
  m_propertyFlags = p;
}



/** Sets the idcode
 */
void GameObject::objectID( IDCode newID )
{
  m_objectID = newID;
}



/** Sets the controller.
 */
void GameObject::setController( BaseController* pController )
{
  delete m_pController;
  m_pController = pController;
}



/** Resurrects this object
 */
void GameObject::resurrect()
{
  this->state( STATE_LIVING );
  this->setHealth( 100 );
  this->hidden( false );
}



/** Manages the messages sent to this object
 */
void GameObject::messagePort( const ObjectMessage& rMessage )
{

  if ( rMessage.m_id == ObjectMessage::KResponse ) {
    return;
  }
  
  // Now we generate a response to the message.
  ObjectMessage responseMessage = ObjectMessage::createResponse( rMessage );
  if ( rMessage.m_pSender != 0 ) {
    rMessage.m_pSender->messagePort( responseMessage );
  }
}



//********************************************************************
//                                                                   *
//      Public GET - methods                                         *
//                                                                   *
//********************************************************************


/** Returns the idcode of this object.
 */
GameObject::IDCode GameObject::objectID() const
{
  return m_objectID;
}


/** Returns the state
 */
GameObject::State GameObject::state() const
{
  return m_state;
}



/** Returns the controlpoints that are rotated by the 'm_angle'
 */
Vec2D GameObject::getCtrlPoint(int pnum) const
{
  WWW_ASSERT( pnum >= 0 );
  WWW_ASSERT( pnum < m_ctrlPoints.size() );
  
  Vec2D tmpV = m_ctrlPoints.at(pnum);
  tmpV.rotate( m_angle );
  return tmpV;
}



/** Returns the value of cnum'th counter
 */
int GameObject::getCounter( int cnum ) const
{
  return m_counters[cnum];
}



/** Returns the healt
 */
int GameObject::health() const
{
  return static_cast<int>( m_health );
}



/** Returns the armorlevel
 */
float GameObject::armor() const
{
  return m_armor;
}



/** Returns the hidden-flag.
 */
bool GameObject::hidden() const
{
  return m_hidden;
}



/** Tells if requested properties are on.
 */
bool GameObject::hasProperty( BProp p ) const
{
  return static_cast<bool>( m_propertyFlags & p );
}



/** Returns all the properties
 */
int GameObject::getProperties() const
{
  return m_propertyFlags;
}



/** Returns the controller.
 */
BaseController* GameObject::getController() const
{
  return m_pController;
}





} // end of namespace
